// Finaquant Analytics - http://finaquant.com/
// Copyright  Tunc Ali Ktkcoglu - Finaquant Analytics GmbH
// related article:
// http://finaquant.com/table-data-as-input-to-estimation-functions-in-r/3082

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Data;
using System.Text.RegularExpressions;
using FinaquantProtos;
using System.Threading;
using RDotNet;

namespace ProtosTest
{
    class Test
    {
		// FunctionRouter with matrix functions in R
        public static void PricePrediction_WithEstimationFunctionsInR()
        {
            #region "Create input tables"

            // define metadata
            MetaData md = MetaData.CreateEmptyMasterData();
            MetaData.AddNewField(md, "country", FieldType.TextAttribute);
            MetaData.AddNewField(md, "product", FieldType.TextAttribute);
            MetaData.AddNewField(md, "year", FieldType.IntegerAttribute);
            MetaData.AddNewField(md, "product_price", FieldType.KeyFigure);
            MetaData.AddNewField(md, "market_index", FieldType.KeyFigure);
            MetaData.AddNewField(md, "oil_price", FieldType.KeyFigure);

            // define global indicator table fields
            var GlobalIndFields = TableFields.CreateEmptyTableFields(md);
            TableFields.AddNewField(GlobalIndFields, "year");
            TableFields.AddNewField(GlobalIndFields, "market_index");
            TableFields.AddNewField(GlobalIndFields, "oil_price");

            // init dictionarues
            var NumAttribValues = new Dictionary<string, NumVector>();
            var TextAttribValues = new Dictionary<string, TextVector>();

            // numeric attribute values
            NumVector YearVal = NumVector.CreateSequenceVector(1993, 1, 20);
            NumAttribValues["year"] = YearVal;

            // default range for all key figures
            KeyValueRange DefaultRangeForAllKeyFigures = KeyValueRange.CreateRange(0, 10);

            // range for selected key figures
            var RangeForSelectedKeywords = new Dictionary<string, KeyValueRange>();
            RangeForSelectedKeywords["oil_price"] = KeyValueRange.CreateRange(60.0, 130.0);
            RangeForSelectedKeywords["market_index"] = KeyValueRange.CreateRange(0.8, 1.5);

            // create indicator table
            var GlobalIndTable = MatrixTable.CombinateFieldValues_B(GlobalIndFields,
                TextAttribValues, NumAttribValues, DefaultRangeForAllKeyFigures, 
                RangeForSelectedKeywords);

            // round all key figures to 2 digits after decimal point
            GlobalIndTable = MatrixTable.Round(GlobalIndTable, 2);

            // view GlobalIndTable
            MatrixTable.View_MatrixTable(GlobalIndTable, "Global Indicators");

            // define table structure
            var PriceTableFields = TableFields.CreateEmptyTableFields(md);
            TableFields.AddNewField(PriceTableFields, "country");
            TableFields.AddNewField(PriceTableFields, "product");
            TableFields.AddNewField(PriceTableFields, "year");
            TableFields.AddNewField(PriceTableFields, "product_price");

            // create test table by combinating field values
            // ... random values for key figures

            // attribute values
            // text attributes
            TextVector CountryVal = TextVector.CreateVectorWithElements("Utopia", "Ignoria", "Euphoria");
            TextVector ProductVal = TextVector.CreateVectorWithElements("bread", "butter", "cheese", "honig");
            
            // assign a value vector to each field
            TextAttribValues["country"] = CountryVal;
            TextAttribValues["product"] = ProductVal;
            NumAttribValues["year"] = YearVal;

            // create test table
            var PriceTable = MatrixTable.CombinateFieldValues_B(PriceTableFields,
                TextAttribValues, NumAttribValues, DefaultRangeForAllKeyFigures, RangeForSelectedKeywords);

            // view PriceTable
            // MatrixTable.View_MatrixTable(PriceTable, "raw PriceTable");

            // price factors for product and country
            var ProductFactorDic = new Dictionary<string, double>();
            ProductFactorDic["bread"] = 1.0;
            ProductFactorDic["cheese"] = 5.0;
            ProductFactorDic["butter"] = 3.0;
            ProductFactorDic["honig"] = 9.0;

            var CountrytFactorDic = new Dictionary<string, double>();
            CountrytFactorDic["Utopia"] = 1.2;
            CountrytFactorDic["Ignoria"] = 1.0;
            CountrytFactorDic["Euphoria"] = 0.8;

            // calculate product prices in price table
            TableRow trow;
            MatrixTable Tout = MatrixTable.CreateEmptyTable(PriceTable.tableFields);
            int year;
            double MarketIndex, OilPrice, ProductFactor, CountryFactor, ProductPrice;

            for (int i = 0; i < PriceTable.RowCount; i++)
            {
                trow = PriceTable.GetTableRow(i);
                year = (int) trow.GetFieldValue("year");

                ProductFactor = ProductFactorDic[(string)trow.GetFieldValue("product")];
                CountryFactor = CountrytFactorDic[(string)trow.GetFieldValue("country")];

                MarketIndex = GlobalIndTable.GetKeyFigValue("market_index", year - 1993);
                OilPrice = GlobalIndTable.GetKeyFigValue("oil_price", year - 1993);

                // formula for product price
                // CountryFactor represents unknown and uncertainty factors
                ProductPrice = ProductFactor * (MarketIndex * 100 / OilPrice) * Math.Sqrt(CountryFactor * 2 + 1);

                // set price
                trow.SetKeyFigure("product_price", ProductPrice);

                // append row to table
                Tout = Tout.AddRowToTable(trow);
            }
            PriceTable = Tout;

            // round all key figures to 2 digits after decimal point
            PriceTable = MatrixTable.Round(PriceTable, 2);

            // view PriceTable
            MatrixTable.View_MatrixTable(PriceTable, "PriceTable");

            #endregion "Create input tables"

            // initialize R connection
            var envPath = Environment.GetEnvironmentVariable("PATH");
            var rBinPath = @"C:\Program Files\R\R-2.14.1\bin\x64";  // check this path on your computer
            Environment.SetEnvironmentVariable("PATH", envPath + System.IO.Path.PathSeparator + rBinPath);

            REngine r_engine = REngine.CreateInstance("RDotNet");
            r_engine.Initialize();

            // initialize price estimation functions in R

            // price estimator A
            string funcstr = @"EstProdPriceFuncA <- function(HistData, EstMarketInd, EstOilPrice) { 
                    RowCount = nrow(HistData);
                    X_train = rbind(matrix(1,1,RowCount), t(HistData[,1:2]));
                    Y_train = matrix(HistData[,3], RowCount, 1);
                    Bopt = solve(X_train %*% t(X_train)) %*% X_train %*% Y_train;
                    X_test = matrix(c(1.0, EstMarketInd, EstOilPrice), nrow=1);
                    Y_test = X_test %*% Bopt;
                    return (Y_test);
                    }";
            Function EstProdPriceFuncA = r_engine.Evaluate(funcstr).AsFunction();

            // price estimator B
            funcstr = @"EstProdPriceFuncB <- function(HistData, EstMarketInd) { 
                RowCount = nrow(HistData);
                X_train = rbind(matrix(1,1,RowCount), t(HistData[,1]));
                Y_train = matrix(HistData[,2], RowCount, 1);
                Bopt = solve(X_train %*% t(X_train)) %*% X_train %*% Y_train;
                X_test = matrix(c(1.0, EstMarketInd), nrow=1);
                Y_test = X_test %*% Bopt;
                return (Y_test);
                }";
            Function EstProdPriceFuncB = r_engine.Evaluate(funcstr).AsFunction();

            // price estimator C
            funcstr = @"EstProdPriceFuncC <- function(HistData, EstOilPrice) { 
                RowCount = nrow(HistData);
                X_train = rbind(matrix(1,1,RowCount), t(HistData[,1]));
                Y_train = matrix(HistData[,2], RowCount, 1);
                Bopt = solve(X_train %*% t(X_train)) %*% X_train %*% Y_train;
                X_test = matrix(c(1.0, EstOilPrice), nrow=1);
                Y_test = X_test %*% Bopt;
                return (Y_test);
                }";
            Function EstProdPriceFuncC = r_engine.Evaluate(funcstr).AsFunction();

            // **************************************************************************************
            // apply table function on a single subtable for country-product pair "Utopia", "bread"
            // **************************************************************************************

            // filter TemperatureTable with condition table to obtain a subtable
            var CondTblFields = TableFields.CreateEmptyTableFields(md);
            TableFields.AddNewField(CondTblFields, "country");
            TableFields.AddNewField(CondTblFields, "product");

            var CondTbl = MatrixTable.CreateTableWithElements_A(CondTblFields,
                "Utopia", "bread"
                );
            var SubTable = MatrixTable.FilterTableA(PriceTable, CondTbl);

            // exclude fields country and product from subtable
            SubTable = MatrixTable.ExcludeColumns(SubTable,
                TextVector.CreateVectorWithElements("country", "product"));

            // view subtable
            // MatrixTable.View_MatrixTable(SubTable, "Subtable of PriceTable (Utopia, bread)");

            // execute user-defined table function to calculate expected energy consumption (Sedrun, 2012)
            MatrixTable EstimatedPriceTbl = PriceEstimatorA(SubTable, GlobalIndTable, 
                r_engine, EstProdPriceFuncA, 1.2, 90.0);

            // view result table
            MatrixTable.View_MatrixTable(EstimatedPriceTbl, "Estimated price for subtable (Utopia, bread)");

            // **************************************************************************************
            // Apply the same table function on all subtables of Price Table
            // **************************************************************************************

            // subtable transformer

            // define subtable fields
            var SubTblFields = TableFields.CreateEmptyTableFields(md);
            TableFields.AddNewField(SubTblFields, "year");
            TableFields.AddNewField(SubTblFields, "product_price");

            // execute subtable transformer
            double EstMarketIndex = 1.2;
            double EstOilPrice = 100.0;

            MatrixTable ResultTbl = MatrixTable.TransformSubTables(PriceTable, SubTblFields,
                PriceEstimatorA, GlobalIndTable, r_engine, EstProdPriceFuncA, EstMarketIndex, EstOilPrice);

            // view result table
            MatrixTable.View_MatrixTable(MatrixTable.Round(ResultTbl,2), 
                "Estimator-A applied for all countries and products");

            // **************************************************************************************
            // Function Router and R
            // **************************************************************************************
            
            // extend MetaData
            md.AddNewField("estimation_method", FieldType.TextAttribute);
            md.AddNewField("country_cell", FieldType.TextAttribute);
            md.AddNewField("product_cell", FieldType.TextAttribute);

            // CASE 1: Apply different price estimators to different countries and products

            // create condition matrix table
            CondTblFields = TableFields.CreateEmptyTableFields(md);
            TableFields.AddNewField(CondTblFields, "country_cell");
            TableFields.AddNewField(CondTblFields, "product_cell");

            var CondMatTbl = MatrixTable.CreateTableWithElements_A(CondTblFields,
                "Utopia", "bread, cheese",      // estimator A
                "Utopia", "butter, honig",      // estimator B
                "Ignoria", "ALL",               // estimator C
                "Euphoria", "ALL"               // estimator A
                );

            // create associated table to see the estimation method in the result table
            var AssocTblFields = TableFields.CreateEmptyTableFields(md);
            TableFields.AddNewField(AssocTblFields, "estimation_method");

            var AssocTbl = MatrixTable.CreateTableWithElements_A(AssocTblFields,
                "estimator A",
                "estimator B",
                "estimator C",
                "estimator A"
                );

            // array for delegate functions; assign a function for each row of condition matrix table
            var MyTableFuncList = new TransformTableFunc_OP[4];
            MyTableFuncList[0] = PriceEstimatorA;
            MyTableFuncList[1] = PriceEstimatorB;
            MyTableFuncList[2] = PriceEstimatorC;
            MyTableFuncList[3] = PriceEstimatorA;

            // array for other parameters
            // pass same parameters for all subtables excluding R Function (estimators A, B, C)
            EstMarketIndex = 1.2;
            EstOilPrice = 100.0;
            
            var OtherParametersList = new object[4][];
            OtherParametersList[0] = new object[] { GlobalIndTable, r_engine, EstProdPriceFuncA, EstMarketIndex, EstOilPrice };
            OtherParametersList[1] = new object[] { GlobalIndTable, r_engine, EstProdPriceFuncB, EstMarketIndex, EstOilPrice };
            OtherParametersList[2] = new object[] { GlobalIndTable, r_engine, EstProdPriceFuncC, EstMarketIndex, EstOilPrice };
            OtherParametersList[3] = new object[] { GlobalIndTable, r_engine, EstProdPriceFuncA, EstMarketIndex, EstOilPrice };

            // call function router C (with condition cell table)
            ResultTbl = MatrixTable.FunctionRouterC(PriceTable, SubTblFields,
                CondMatTbl, MyTableFuncList, AssocTbl, FirstMatchOnly: true, IgnoreHierarchy: true,
                ErrorIfConditionNotRelevant: false, OtherParametersList: OtherParametersList);

            // round all key figures to 2 digits after decimal point
            ResultTbl = MatrixTable.Round(ResultTbl, 2);

            // view result table
            MatrixTable.View_MatrixTable(ResultTbl, "Result table: Apply different estimators for different subtables");

            // CASE 2: Apply all price estimators to all countries and products

            // create condition table (not condition matrix table with condition cells!)
            CondTblFields = TableFields.CreateEmptyTableFields(md);
            TableFields.AddNewField(CondTblFields, "country");

            CondTbl = MatrixTable.CreateTableWithElements_A(CondTblFields,
                "ALL",                          // estimator A
                "ALL",                          // estimator B
                "ALL"                           // estimator C
                );

            // create associated table to see the estimation method in the result table
            AssocTbl = MatrixTable.CreateTableWithElements_A(AssocTblFields,
                "estimator A",
                "estimator B",
                "estimator C"
                );

            // array for delegate functions; assign a function for each row of condition matrix table
            MyTableFuncList = new TransformTableFunc_OP[3];
            MyTableFuncList[0] = PriceEstimatorA;
            MyTableFuncList[1] = PriceEstimatorB;
            MyTableFuncList[2] = PriceEstimatorC;

            // array for other parameters
            // pass same parameters for all subtables excluding R Function (estimators A, B, C)
            EstMarketIndex = 1.2;
            EstOilPrice = 100.0;

            OtherParametersList = new object[3][];
            OtherParametersList[0] = new object[] { GlobalIndTable, r_engine, EstProdPriceFuncA, EstMarketIndex, EstOilPrice };
            OtherParametersList[1] = new object[] { GlobalIndTable, r_engine, EstProdPriceFuncB, EstMarketIndex, EstOilPrice };
            OtherParametersList[2] = new object[] { GlobalIndTable, r_engine, EstProdPriceFuncC, EstMarketIndex, EstOilPrice };

            // call function router A (with condition table)
            ResultTbl = MatrixTable.FunctionRouterA(PriceTable, SubTblFields,
                CondTbl, MyTableFuncList, AssocTbl, 
                JokerMatchesAllvalues: true, TextJoker: "ALL", NumJoker: 0,
                FirstMatchOnly: false, OtherParametersList: OtherParametersList);

            // round all key figures to 2 digits after decimal point
            ResultTbl = MatrixTable.Round(ResultTbl, 2);

            // view result table
            MatrixTable.View_MatrixTable(ResultTbl, "Result table: Apply all estimators for all subtables");
        }

        // Table function for price estimation, estimator A (LR based on global market index and oil price)
        // Input parameters:
        // InputTbl: Historical product prices with fields year and price
        // OtherParameters[0]: Table of historical indicators with fields year, market_index and oil_price
        // OtherParameters[1]: REngine
        // OtherParameters[2]: R Function
        // OtherParameters[3]: Estimated market index for next year (double)
        // OtherParameters[4]: Estimated oil price for next year (double)
        public static MatrixTable PriceEstimatorA(MatrixTable InputTbl, params Object[] OtherParameters)
        {
            // get all input parameters
            MatrixTable HistoricalProductPrices = InputTbl;
            MatrixTable HistoricalIndicators = (MatrixTable)OtherParameters[0];
            REngine engine = (REngine)OtherParameters[1];
            Function EstProdPriceFuncA = (Function)OtherParameters[2];
            double[] EstMarketInd = {(double) OtherParameters[3]};
            double[] EstOilPrice = {(double) OtherParameters[4]};

            // combine tables
            MatrixTable HistoricalDataTbl = MatrixTable.CombineTables(HistoricalIndicators, HistoricalProductPrices);

            // get matrix of key figures from table
            KeyMatrix HistData = HistoricalDataTbl.KeyFigValues;

            // convert data types of input parameters from C# to R
            NumericMatrix HistDataR = engine.CreateNumericMatrix(HistData.toArray);
            engine.SetSymbol("HistDataR", HistDataR);

            NumericVector EstMarketIndR = engine.CreateNumericVector(EstMarketInd);
            engine.SetSymbol("EstMarketIndR", EstMarketIndR);

            NumericVector EstOilPriceR = engine.CreateNumericVector(EstOilPrice);
            engine.SetSymbol("EstOilPriceR", EstOilPriceR);

            // TEST
            NumericVector x = engine.Evaluate(@"x <- nrow(HistDataR)").AsNumeric();
            System.Diagnostics.Debug.WriteLine("nrow(HistDataR): " + x[0]);

            // call function in R from c#
            NumericMatrix EstProductPrice = engine.Evaluate(
            @"EstProductPrice <- EstProdPriceFuncA(HistDataR,EstMarketIndR,EstOilPriceR)").AsNumericMatrix();

            // return a single-element table with estimated product price
            double EstProdPrice = EstProductPrice[0, 0];

            // return a single-element table with estimated product price
            TableFields tf = new TableFields(InputTbl.metaData);
            tf.AddNewField("product_price");
            MatrixTable ResultTbl = MatrixTable.CreateTableWithElements_A(tf, EstProdPrice);

            // ResultTbl = MatrixTable.CreateTableWithElements_A(tf, 2.5);
            return ResultTbl;
        }

        // Table function for price estimation, estimator B (LR based on global market index only)
        // Input parameters:
        // InputTbl: Historical product prices with fields year and price
        // OtherParameters[0]: Table of historical indicators with fields year, market_index and oil_price
        // OtherParameters[1]: REngine
        // OtherParameters[2]: R Function
        // OtherParameters[3]: Estimated market index for next year (double)
        // OtherParameters[4]: Estimated oil price for next year (double)
        public static MatrixTable PriceEstimatorB(MatrixTable InputTbl, params Object[] OtherParameters)
        {
            // get all input parameters
            MatrixTable HistoricalProductPrices = InputTbl;
            MatrixTable HistoricalIndicators = (MatrixTable)OtherParameters[0];
            REngine engine = (REngine)OtherParameters[1];
            Function EstProdPriceFuncB = (Function)OtherParameters[2];
            double[] EstMarketInd = { (double)OtherParameters[3] };
            double[] EstOilPrice = { (double)OtherParameters[4] };

            // combine tables
            MatrixTable HistoricalDataTbl = MatrixTable.CombineTables(HistoricalIndicators, HistoricalProductPrices);

            // exclude column oil_price from table
            HistoricalDataTbl = MatrixTable.ExcludeColumns(HistoricalDataTbl, 
                TextVector.CreateVectorWithElements("oil_price"));

            // get matrix of key figures from table
            KeyMatrix HistData = HistoricalDataTbl.KeyFigValues;

            // convert data types of input parameters from C# to R
            NumericMatrix HistDataR = engine.CreateNumericMatrix(HistData.toArray);
            engine.SetSymbol("HistDataR", HistDataR);

            NumericVector EstMarketIndR = engine.CreateNumericVector(EstMarketInd);
            engine.SetSymbol("EstMarketIndR", EstMarketIndR);

            NumericVector EstOilPriceR = engine.CreateNumericVector(EstOilPrice);
            engine.SetSymbol("EstOilPriceR", EstOilPriceR);

            // call function in R from c#
            NumericMatrix EstProductPrice = engine.Evaluate(
            @"EstProductPrice <- EstProdPriceFuncB(HistDataR,EstMarketIndR)").AsNumericMatrix();

            // return a single-element table with estimated product price
            double EstProdPrice = EstProductPrice[0, 0];

            // return a single-element table with estimated product price
            TableFields tf = new TableFields(InputTbl.metaData);
            tf.AddNewField("product_price");
            MatrixTable ResultTbl = MatrixTable.CreateTableWithElements_A(tf, EstProdPrice);
            return ResultTbl;
        }

        // Table function for price estimation, estimator C (LR based on global oil price only)
        // Input parameters:
        // InputTbl: Historical product prices with fields year and price
        // OtherParameters[0]: Table of historical indicators with fields year, market_index and oil_price
        // OtherParameters[1]: REngine
        // OtherParameters[2]: R Function
        // OtherParameters[3]: Estimated market index for next year (double)
        // OtherParameters[4]: Estimated oil price for next year (double)
        public static MatrixTable PriceEstimatorC(MatrixTable InputTbl, params Object[] OtherParameters)
        {
            // get all input parameters
            MatrixTable HistoricalProductPrices = InputTbl;
            MatrixTable HistoricalIndicators = (MatrixTable)OtherParameters[0];
            REngine engine = (REngine)OtherParameters[1];
            Function EstProdPriceFuncC = (Function)OtherParameters[2];
            double[] EstMarketInd = { (double)OtherParameters[3] };
            double[] EstOilPrice = { (double)OtherParameters[4] };

            // combine tables
            MatrixTable HistoricalDataTbl = MatrixTable.CombineTables(HistoricalIndicators, HistoricalProductPrices);

            // exclude column market_index from table
            HistoricalDataTbl = MatrixTable.ExcludeColumns(HistoricalDataTbl,
                TextVector.CreateVectorWithElements("market_index"));

            // get matrix of key figures from table
            KeyMatrix HistData = HistoricalDataTbl.KeyFigValues;

            // convert data types of input parameters from C# to R
            NumericMatrix HistDataR = engine.CreateNumericMatrix(HistData.toArray);
            engine.SetSymbol("HistDataR", HistDataR);

            NumericVector EstMarketIndR = engine.CreateNumericVector(EstMarketInd);
            engine.SetSymbol("EstMarketIndR", EstMarketIndR);

            NumericVector EstOilPriceR = engine.CreateNumericVector(EstOilPrice);
            engine.SetSymbol("EstOilPriceR", EstOilPriceR);

            // call function in R from c#
            NumericMatrix EstProductPrice = engine.Evaluate(
            @"EstProductPrice <- EstProdPriceFuncC(HistDataR,EstOilPriceR)").AsNumericMatrix();

            // return a single-element table with estimated product price
            double EstProdPrice = EstProductPrice[0, 0];

            // return a single-element table with estimated product price
            TableFields tf = new TableFields(InputTbl.metaData);
            tf.AddNewField("product_price");
            MatrixTable ResultTbl = MatrixTable.CreateTableWithElements_A(tf, EstProdPrice);
            return ResultTbl;
        }
	}
}